<?php
/**
 * Blocksy functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package Blocksy
 */

if (version_compare(PHP_VERSION, '5.7.0', '<')) {
    require get_template_directory() . '/inc/php-fallback.php';
    return;
}

require get_template_directory() . '/inc/init.php';

function enqueue_breaking_news_slider_assets()
{
    wp_enqueue_style(
        'breaking-news-slider-styles',
        get_template_directory_uri() . '/style.css'
    );

    wp_enqueue_script(
        'breaking-news-slider-script',
        get_template_directory_uri() . '/custom.js',
        array('jquery'),
        null,
        true
    );
}
add_action('wp_enqueue_scripts', 'enqueue_breaking_news_slider_assets');

function custom_mime_types($mimes) {
    // Add new MIME types
    $mimes['svg'] = 'image/svg+xml';
    $mimes['woff'] = 'font/woff';
    $mimes['woff2'] = 'font/woff2';

    return $mimes;
}
add_filter('upload_mimes', 'custom_mime_types');

// Add a custom shortcode for the Polylang language switcher
function enqueue_select2_for_language_switcher()
{
    wp_enqueue_style('select2-css', 'https://cdnjs.cloudflare.com/ajax/libs/select2/4.0.13/css/select2.min.css');
    wp_enqueue_script('select2-js', 'https://cdnjs.cloudflare.com/ajax/libs/select2/4.0.13/js/select2.min.js', array('jquery'), null, true);
    wp_add_inline_script('select2-js', "
        jQuery(document).ready(function($) {
            $('#language-switcher').select2({
                minimumResultsForSearch: -1, // Disable search
                width: 'auto', // Adjust width automatically
                templateResult: formatLanguage, // Format dropdown
                templateSelection: formatLanguage
            });

            function formatLanguage(lang) {
                if (!lang.id) return lang.text;
                return $('<span><span style=\"margin-right:5px;\">🌐</span>' + lang.text + '</span>');
            }
        });
    ");
}
add_action('wp_enqueue_scripts', 'enqueue_select2_for_language_switcher');

// Add a custom shortcode for the Polylang language switcher
function custom_polylang_select_with_icon()
{
    if (!function_exists('pll_the_languages')) {
        return '<p>Polylang is not activated.</p>';
    }

    $languages = pll_the_languages(array(
        'echo' => 0,
        'raw' => 1,
    ));

    if (empty($languages)) {
        return '';
    }

    $output = '<div class="lang-select-wrapper">';
    $output .= '<select onchange="if(this.value) window.location.href=this.value;">';

    foreach ($languages as $lang) {
        $selected = $lang['current_lang'] ? 'selected' : '';
        $output .= '<option value="' . esc_url($lang['url']) . '" ' . $selected . '>';
        $output .= strtoupper($lang['slug']);
        $output .= '</option>';
    }

    $output .= '</select>';
    $output .= '</div>';

    return $output;
}
add_shortcode('lang_select_icon', 'custom_polylang_select_with_icon');

// Add a custom shortcode for the light/dark mode switcher
function custom_light_dark_switcher() {
    ob_start();
    $unique_id = uniqid();
    ?>
    <div class="theme-switcher">
        <?php
        $upload_dir = wp_upload_dir();
        $sun_icon = $upload_dir['baseurl'] . "/2025/02/sun-03-stroke-rounded.svg";
        $moon_icon = $upload_dir['baseurl'] . "/2025/02/moon-02-stroke-rounded-1.svg";
        ?>
        <input type="checkbox" class="theme-toggle" id="theme-toggle-<?php echo $unique_id; ?>">
        <label class="toggle-label" for="theme-toggle-<?php echo $unique_id; ?>">
            <img src="<?php echo esc_url($sun_icon); ?>" alt="Sun" class="sun-icon" width="20" height="20">
            <img src="<?php echo esc_url($moon_icon); ?>" alt="Moon" class="moon-icon" width="20" height="20">
        </label>
    </div>
    <?php
    // Only include the script once
    static $script_printed = false;
    if (!$script_printed) {
        $script_printed = true;
        ?>
        <script>
            document.addEventListener('DOMContentLoaded', function () {
                const switches = document.querySelectorAll('.theme-toggle');
                const body = document.body;

                // Check for saved theme preference or default to 'light'
                let currentTheme = localStorage.getItem('theme') || 'light';

                function updateVisualState(isDark) {
                    // Update all switches
                    switches.forEach(sw => sw.checked = isDark);

                    // Apply theme to body
                    if (isDark) {
                        body.classList.add('dark-mode');
                    } else {
                        body.classList.remove('dark-mode');
                    }
                }

                // Apply initial theme
                updateVisualState(currentTheme === 'dark');

                // Add event listeners to all switches
                switches.forEach(sw => {
                    sw.addEventListener('change', function () {
                        const isDark = this.checked;
                        const theme = isDark ? 'dark' : 'light';

                        // Save theme preference
                        localStorage.setItem('theme', theme);

                        // Update visual state
                        updateVisualState(isDark);

                        // Sync all other switches
                        switches.forEach(other => {
                            if (other !== this) {
                                other.checked = isDark;
                            }
                        });
                    });
                });
            });
        </script>
        <?php
    }

    return ob_get_clean();
}

add_shortcode('light_dark_switcher', 'custom_light_dark_switcher');

// Add a custom shortcode for the breaking news slider
function breaking_news_slider_shortcode($atts)
{
    $atts = shortcode_atts(
        array(
            'posts' => 5,
        ),
        $atts,
        'breaking_news_slider'
    );

    $query_args = array(
        'category_name' => 'breaking-news',
        'posts_per_page' => intval($atts['posts']),
    );

    $query = new WP_Query($query_args);
    $output = '<div class="breaking-news-slider"><div class="slider-track">';

    if ($query->have_posts()) {
        while ($query->have_posts()) {
            $query->the_post();

            $output .= '<div class="slider-slide">
                        <a href="' . get_permalink() . '">
                            ' . get_the_post_thumbnail(get_the_ID(), 'large', array('class' => 'slider-image')) . '
                        </a>
                        <div class="slider-content">
                            <span class="slider-tag">Breaking News</span>
                            <h3><a href="' . get_permalink() . '">' . get_the_title() . '</a></h3>
                            <p>' . wp_trim_words(get_the_excerpt(), 15) . '</p>
                        </div>
                    </div>';
        }

        $output .= '<div class="slider-dots">';
        for ($i = 0; $i < $query->post_count; $i++) {
            $activeClass = $i === 0 ? 'active' : '';
            $output .= '<span class="' . $activeClass . '" data-slide="' . $i . '"></span>';
        }
        $output .= '</div></div>';

        $output .= '<script>
    document.addEventListener("DOMContentLoaded", function () {
        const slides = document.querySelectorAll(".slider-slide");
        const dots = document.querySelectorAll(".slider-dots span");
        let currentIndex = 0;
        let interval = null;

        function updateActiveSlide(index) {
            slides.forEach((slide, i) => {
                slide.style.display = i === index ? "block" : "none";
            });

            dots.forEach((dot, i) => {
                dot.classList.toggle("active", i === index);
            });
        }

        function startAutoplay() {
            interval = setInterval(() => {
                currentIndex = (currentIndex + 1) % slides.length;
                updateActiveSlide(currentIndex);
            }, 4000); // 4000ms = 4 seconds
        }

        function stopAutoplay() {
            clearInterval(interval);
        }

        dots.forEach((dot, index) => {
            dot.addEventListener("click", () => {
                currentIndex = index;
                updateActiveSlide(currentIndex);
                stopAutoplay();
                startAutoplay(); // restart autoplay after manual click
            });
        });

        updateActiveSlide(currentIndex);
        startAutoplay();
    });
</script>';
    } else {
        $output .= '<p>No recent posts found.</p>';
    }

    $output .= '</div>';

    wp_reset_postdata();

    return $output;
}
add_shortcode('breaking_news_slider', 'breaking_news_slider_shortcode');

// Add a custom shortcode for the recent posts card
function recent_posts_card_shortcode($atts)
{
    $atts = shortcode_atts(
        array(
            'posts' => 3,
        ),
        $atts,
        'recent_posts'
    );

    $query_args = array(
        'post_type' => 'post',
        'posts_per_page' => intval($atts['posts']),
    );

    $query = new WP_Query($query_args);

    $output = '<div class="recent-posts-cards">';

    if ($query->have_posts()) {
        while ($query->have_posts()) {
            $query->the_post();

            // Get post category
            $categories = get_the_category();
            $category_name = !empty($categories) ? esc_html($categories[0]->name) : 'Uncategorized';

            // Get post time
            $post_time = 'Postat: acum ' . human_time_diff(get_the_time('U'), current_time('timestamp'));

            $output .= '<div class="post-card">
                <a href="' . get_permalink() . '" class="post-card-image-link">
                    ' . get_the_post_thumbnail(get_the_ID(), 'medium', array('class' => 'post-card-image')) . '
                </a>
                <div class="post-card-content">
                    <h3 class="post-card-title"><a href="' . get_permalink() . '">' . get_the_title() . '</a></h3>
                    <div class="post-meta">
                        <span class="post-category">
                            <i class="hgi hgi-stroke hgi-grid"></i>
                            ' . $category_name . '
                        </span>
                        <span class="separator"> | </span>
                        <span class="post-time">
                            ' . $post_time . '
                        </span>
                    </div>
                </div>
            </div>';
        }
    } else {
        $output .= '<p>No recent posts found.</p>';
    }

    $output .= '</div>';

    wp_reset_postdata();

    return $output;
}
add_shortcode('recent_posts', 'recent_posts_card_shortcode');

function apple_unlocked_articles_shortcode($atts)
{
    // Attributes
    $atts = shortcode_atts(array(
        'category' => 'apple-unlocked', // slug of the category
        'posts_per_page' => 3,
    ), $atts);

    // WP Query
    $query = new WP_Query(array(
        'category_name' => $atts['category'],
        'posts_per_page' => $atts['posts_per_page'],
    ));

    // Start output
    ob_start();

    if ($query->have_posts()) {
        echo '<div class="apple-unlocked-articles">';

        while ($query->have_posts()) {
            $query->the_post();

            echo '<div class="article-card">';
            echo get_the_post_thumbnail(get_the_ID(), 'medium');
            echo '<div class="article-content">';
            echo '<span class="badge"><i class="hgi hgi-stroke hgi-calendar-03"></i> ' . get_the_date('j F Y') . '</span>';
            echo '<h3>' . get_the_title() . '</h3>';
            echo '<p>' . get_the_excerpt() . '</p>';
            echo '<a href="' . get_the_permalink() . '" class="read-more"><i class="hgi hgi-stroke hgi-book-open-01"></i> Citește mai mult</a>';
            echo '</div>'; // .article-content
            echo '</div>'; // .article-card
        }

        echo '</div>'; // .apple-unlocked-articles

        wp_reset_postdata();
    } else {
        echo '<p>No articles found.</p>';
    }

    return ob_get_clean();
}
add_shortcode('apple_articles', 'apple_unlocked_articles_shortcode');

/* Custom comments template */
function my_custom_comment_form()
{
    $fields = [
        'author' => '',
        'email' => '',
        'url' => ''
    ];

    $args = [
        'fields' => apply_filters('comment_form_default_fields', $fields),

        'comment_field' => '
            <div class="form-group">
                <textarea id="comment" name="comment" class="form-control" placeholder="Scrie comentariul aici..." rows="6" required></textarea>
            </div>',

        'title_reply' => '<div class="form-title"><i class="hgi hgi-stroke hgi-bubble-chat"></i> LASĂ UN COMENTARIU</div>',

        'label_submit' => 'Trimite',
        'submit_button' => '<button type="submit" class="comment-submit"><span>Trimite</span> <i class="hgi hgi-stroke hgi-sent"></i></button>',

        'comment_notes_before' => '',
        'comment_notes_after' => '',

        'logged_in_as' => '',
    ];

    comment_form($args);
}

function shortcode_portfolio_slider($atts)
{
    $atts = shortcode_atts(array(
        'posts_per_page' => 6,
    ), $atts);

    $query = new WP_Query(array(
        'post_type' => 'projects',
        'posts_per_page' => $atts['posts_per_page'],
    ));

    ob_start();

    if ($query->have_posts()) {
        ?>
        <!-- Swiper Container -->
        <div class="swiper portfolio-swiper">
            <div class="swiper-wrapper">
                <?php while ($query->have_posts()):
                    $query->the_post(); ?>
                    <div class="swiper-slide">
                        <div class="portfolio-card">
                            <?php if (has_post_thumbnail()): ?>
                                <div class="portfolio-thumb">
                                    <?php the_post_thumbnail('medium'); ?>
                                </div>
                            <?php endif; ?>
                            <div class="portfolio-content">
                                <!-- <p class="portfolio-category"><?php the_category(', '); ?></p> -->
                                <?php
                                $terms = get_the_terms(get_the_ID(), 'project_category');
                                if (!empty($terms) && !is_wp_error($terms)) {
                                    echo '<p class="portfolio-category">';
                                    $term_names = wp_list_pluck($terms, 'name');
                                    echo esc_html(implode(', ', $term_names));
                                    echo '</p>';
                                }
                                ?>
                                <h3 class="portfolio-title"><?php the_title(); ?></h3>
                                <p class="portfolio-excerpt"><?php echo get_the_excerpt(); ?></p>
                                <?php
                                $project_url = get_field('project_url');
                                ?>
                                <?php if ($project_url): ?>
                                    <a href="<?php echo esc_url($project_url); ?>" class="portfolio-link" target="_blank"
                                        rel="noopener">
                                        <i class="hgi hgi-stroke hgi-link-square-01"></i> Demo Live
                                    </a>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                <?php endwhile; ?>
            </div>
        </div>

        <script>
            document.addEventListener("DOMContentLoaded", function () {
                new Swiper(".portfolio-swiper", {
                    slidesPerView: 1,
                    spaceBetween: 30,
                    loop: true,
                    autoplay: {
                        delay: 3000,
                        disableOnInteraction: false,
                    },
                    pagination: {
                        el: ".swiper-pagination",
                        clickable: true,
                    },
                    navigation: {
                        nextEl: ".swiper-button-next",
                        prevEl: ".swiper-button-prev",
                    },
                    breakpoints: {
                        768: { slidesPerView: 2 },
                        1025: { slidesPerView: 3 }
                    }
                });
            });
        </script>
        <?php
        wp_reset_postdata();
    } else {
        echo '<p>Nu există proiecte în portofoliu.</p>';
    }

    return ob_get_clean();
}
add_shortcode('portfolio_slider', 'shortcode_portfolio_slider');

function shortcode_portfolio_slider_nav()
{
    ob_start();
    ?>
    <div class="portfolio-slider-nav">
        <div class="swiper-button-prev custom-nav-button">
            <span>&larr;</span>
        </div>
        <div class="swiper-button-next custom-nav-button">
            <span>&rarr;</span>
        </div>
    </div>
    <?php
    return ob_get_clean();
}
add_shortcode('portfolio_slider_nav', 'shortcode_portfolio_slider_nav');

function enqueue_related_swiper_assets()
{
    // Swiper CSS
    wp_enqueue_style(
        'swiper-css',
        'https://cdn.jsdelivr.net/npm/swiper@11/swiper-bundle.min.css',
        [],
        null
    );

    // Swiper JS
    wp_enqueue_script(
        'swiper-js',
        'https://cdn.jsdelivr.net/npm/swiper@11/swiper-bundle.min.js',
        [],
        null,
        true
    );

    // Custom script to initialize Swiper
    wp_add_inline_script('swiper-js', "
    document.addEventListener('DOMContentLoaded', function () {
      const swiper = new Swiper('.related-swiper', {
        slidesPerView: 1,
        spaceBetween: 37,
        loop: true,
        pagination: {
          el: '.swiper-pagination',
          clickable: true,
        },
        breakpoints: {
          768: { slidesPerView: 2 },
          1025: { slidesPerView: 3 }
        }
      });
    });
  ");
}
add_action('wp_enqueue_scripts', 'enqueue_related_swiper_assets');

function shortcode_portfolio_tabs_grid($atts)
{
    $atts = shortcode_atts(array(
        'posts_per_page' => 12,
    ), $atts);

    $paged = get_query_var('paged') ? get_query_var('paged') : 1;

    // Get all categories
    $terms = get_terms(array(
        'taxonomy' => 'project_category',
        'hide_empty' => true,
    ));

    // Query projects paginated
    $projects = new WP_Query(array(
        'post_type' => 'projects',
        'posts_per_page' => $atts['posts_per_page'],
        'post_status' => 'publish',
        'paged' => $paged,
    ));

    ob_start();
    ?>

    <!-- Tabs -->
    <?php if (!empty($terms) && !is_wp_error($terms)): ?>
        <div class="portfolio-tabs">
            <button class="portfolio-tab active" data-filter="all">Toate</button>
            <?php foreach ($terms as $term): ?>
                <button class="portfolio-tab" data-filter="<?php echo esc_attr($term->slug); ?>">
                    <?php echo esc_html($term->name); ?>
                </button>
            <?php endforeach; ?>
        </div>
    <?php endif; ?>

    <!-- Grid -->
    <div class="portfolio-grid">
        <?php if ($projects->have_posts()): ?>
            <?php while ($projects->have_posts()):
                $projects->the_post();
                $term_list = wp_get_post_terms(get_the_ID(), 'project_category', array('fields' => 'slugs'));
                $categories = is_wp_error($term_list) ? '' : implode(' ', $term_list);
                ?>
                <div class="portfolio-card" data-category="<?php echo esc_attr($categories); ?>">
                    <?php if (has_post_thumbnail()): ?>
                        <div class="portfolio-thumb">
                            <?php the_post_thumbnail('medium'); ?>
                        </div>
                    <?php endif; ?>
                    <div class="portfolio-content">
                        <?php
                        $terms = get_the_terms(get_the_ID(), 'project_category');
                        if (!empty($terms) && !is_wp_error($terms)) {
                            echo '<p class="portfolio-category">';
                            $term_names = wp_list_pluck($terms, 'name');
                            echo esc_html(implode(', ', $term_names));
                            echo '</p>';
                        }
                        ?>
                        <h3 class="portfolio-title"><?php the_title(); ?></h3>
                        <p class="portfolio-excerpt"><?php echo get_the_excerpt(); ?></p>
                        <?php $project_url = get_field('project_url'); ?>
                        <?php if ($project_url): ?>
                            <a href="<?php echo esc_url($project_url); ?>" class="portfolio-link" target="_blank" rel="noopener">
                                <i class="hgi hgi-stroke hgi-link-square-01"></i> Demo Live
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endwhile; ?>
            <?php wp_reset_postdata(); ?>
        <?php else: ?>
            <p>Nu există proiecte disponibile.</p>
        <?php endif; ?>
    </div>

    <!-- Pagination -->
    <div class="portfolio-pagination">
        <?php
        echo paginate_links(array(
            'total' => $projects->max_num_pages,
            'current' => $paged,
            'prev_text' => __('« Înapoi'),
            'next_text' => __('Înainte »'),
        ));
        ?>
    </div>

    <!-- JS Filtering (unchanged) -->
    <script>
        document.addEventListener("DOMContentLoaded", function () {
            const tabs = document.querySelectorAll(".portfolio-tab");
            const items = document.querySelectorAll(".portfolio-card");

            tabs.forEach(tab => {
                tab.addEventListener("click", () => {
                    tabs.forEach(t => t.classList.remove("active"));
                    tab.classList.add("active");

                    const filter = tab.getAttribute("data-filter");

                    items.forEach(item => {
                        const itemCats = item.getAttribute("data-category").split(' ');
                        if (filter === "all" || itemCats.includes(filter)) {
                            item.style.display = "block";
                        } else {
                            item.style.display = "none";
                        }
                    });
                });
            });
        });
    </script>

    <style>
        .portfolio-pagination {
            margin-top: 2rem;
            text-align: center;
        }

        .portfolio-pagination .page-numbers {
            margin: 0 5px;
            padding: 6px 12px;
            background: #eee;
            color: #333;
            text-decoration: none;
            border-radius: 4px;
        }

        .portfolio-pagination .current {
            background: #000;
            color: #fff;
        }
    </style>

    <?php
    return ob_get_clean();
}
add_shortcode('portfolio_tabs_grid', 'shortcode_portfolio_tabs_grid');